<?php

/**
 * Creates the Panel of share buttons that appear on the side of the screen and
 * floats as the user scrolls the page based on options and settings.
 *
 * @package   SocialWarfare\Functions
 * @copyright Copyright (c) 2018, Warfare Plugins, LLC
 * @license   GPL-3.0+
 * @since     3.4.0 | 21 SEP 2018 | Moved from Buttons_Panel into its own child
 *                                  class extending Buttons Panel.
 */
class SWP_Buttons_Panel_Side extends SWP_Buttons_Panel {


	/**
	 * This is the type of buttons panel being generated by this class. The
	 * parent class uses static_horizontal.
	 */
	public $panel_type = 'floating_side';


	/**
	 * Instantiate the object and prepare to have a buttons panel rendered. The
	 * primary purpose of having a constructor in this child class is so that we
	 * can add the $max_buttons property to the object.
	 *
	 * @since  3.4.0 | 24 OCT 2018 | Created
	 * @param  void
	 * @return void
	 */
	public function __construct() {
		parent::__construct();

		/**
		 * The floating button doesn't look good if you have a ton of buttons
		 * enabled. As such, we have an option to limit the number of buttons
		 * displayed. This is the maximum.
		 */
		$max_buttons = $this->get_option( 'float_button_count' );
		if ( false === $max_buttons || 0 === $max_buttons ) {
			$max_buttons = 5;
		}
		$this->max_buttons = $max_buttons;
	}


	/**
	 * A method to determine whether or not the panel should be generated and
	 * displayed on this page. This overrides a method of the same name in the
	 * parent class.
	 *
	 * @since  3.4.0 | 24 OCT 2018 | Created
	 * @param  void
	 * @return bool true to display it; false to not display it.
	 */
	public function should_panel_display() {

		/**
		 * If for some reason the post_data failed to populate, we just have to
		 * bail out so the PHP doesn't throw undefined property errors.
		 */
		if ( empty( $this->post_data ) ) {
			return false;
		}

		/**
		 * We are only generating the floating buttons panel if it is set to
		 * left or right. If it is set to none, top, or bottom, we don't need it.
		 */
		$blacklist = array( 'none', 'top', 'bottom' );
		if ( in_array( $this->get_option( 'float_location' ), $blacklist, true ) ) {
			return false;
		}

		if ( in_array( $this->get_float_location(), $blacklist, true ) ) {
			return false;
		}

		/**
		 * Bail out if we're not on a single.php, if the floating buttons are
		 * turned off, or if this is a post preview.
		 */
		if ( ! is_singular() || is_admin() || is_feed() || is_search() || is_attachment() || is_preview() ) {
			return false;
		}

		return true;
	}


	/**
	 * Generate the CSS classes for the parent wrapper container. This overrides
	 * a method of the same name in the parent class.
	 *
	 * @since  3.4.0 | 21 SEP 2018 | Created
	 * @param  void
	 * @return string A string of CSS classes.
	 */
	protected function generate_css_classes() {

		$classes  = 'class="';
		$classes .= 'swp_social_panelSide swp_floating_panel swp_social_panel';
		$classes .= ' swp_' . $this->get_option( 'float_button_shape' );
		$classes .= $this->get_colors( true );
		$classes .= $this->get_option( 'transition' );

		/**
		 * This controls whether the floating panel is going to be displayed on
		 * the left side of the screen or the right side of the screen.
		 */
		if ( 'none' !== $this->get_float_location() ) {
			$classes .= ' swp_float_' . $this->get_option( 'float_location' );
		}

		/**
		 * This determines if the floating buttons will be snug against the top
		 * of the screen, the bottom of the screen or centered vertically.
		 */
		if ( $this->get_option( 'float_alignment' ) ) {
			$classes .= ' swp_side_' . $this->get_option( 'float_alignment' );
		}

		/**
		 * This determines if the user has set the size of the panel. If so, the
		 * CSS will use the transform:scale() to make it that size.
		 */
		if ( isset( $this->options['float_size'] ) ) {
			$size     = $this->get_option( 'float_size' ) * 100;
			$side     = $this->get_option( 'float_location' );
			$position = $this->get_option( 'float_alignment' );
			$classes .= " scale-{$size} float-position-{$position}-{$side}";
		}

		$classes .= '" ';

		$this->classes = $classes;
	}
}
